/*
 * Copyright 2009-2024 PrimeTek.
 *
 * https://www.primefaces.org/lts/licenses/
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.primefaces.selenium.internal;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;

import org.openqa.selenium.PageLoadStrategy;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.chrome.ChromeDriver;
import org.openqa.selenium.chrome.ChromeDriverLogLevel;
import org.openqa.selenium.chrome.ChromeOptions;
import org.openqa.selenium.firefox.FirefoxDriver;
import org.openqa.selenium.firefox.FirefoxDriverLogLevel;
import org.openqa.selenium.firefox.FirefoxOptions;
import org.openqa.selenium.logging.LogType;
import org.openqa.selenium.logging.LoggingPreferences;
import org.openqa.selenium.safari.SafariDriver;
import org.openqa.selenium.safari.SafariOptions;
import org.primefaces.selenium.spi.WebDriverAdapter;

import io.github.bonigarcia.wdm.WebDriverManager;

public class DefaultWebDriverAdapter implements WebDriverAdapter {

    @Override
    public void initialize(ConfigProvider configProvider) {
        WebDriverManager webDriverManager = null;
        if ("firefox".equals(configProvider.getWebdriverBrowser())) {
            if (!System.getProperties().contains("webdriver.gecko.driver")) {
                webDriverManager = WebDriverManager.firefoxdriver();
            }
        }
        else if ("chrome".equals(configProvider.getWebdriverBrowser())) {
            if (!System.getProperties().contains("webdriver.chrome.driver")) {
                webDriverManager = WebDriverManager.chromedriver();
            }
        }
        else if ("safari".equals(configProvider.getWebdriverBrowser())) {
            if (!System.getProperties().contains("webdriver.safari.driver")) {
                webDriverManager = WebDriverManager.safaridriver();
            }
        }

        if (webDriverManager != null) {
            if (configProvider.getWebdriverVersion() != null) {
                webDriverManager = webDriverManager.driverVersion(configProvider.getWebdriverVersion());
            }

            webDriverManager.setup();
        }
    }

    @Override
    public WebDriver createWebDriver() {
        ConfigProvider config = ConfigProvider.getInstance();
        if (config.getWebdriverBrowser() == null) {
            throw new RuntimeException("No webdriver.browser configured; Please either configure it or implement WebDriverAdapter#getWebDriver!");
        }

        LoggingPreferences logPrefs = new LoggingPreferences();
        logPrefs.enable(LogType.BROWSER, Level.ALL);

        switch (config.getWebdriverBrowser()) {
            case "firefox":
                FirefoxOptions firefoxOptions = new FirefoxOptions().configureFromEnv();
                firefoxOptions.setPageLoadStrategy(PageLoadStrategy.NORMAL);
                if (config.isWebdriverHeadless()) {
                    firefoxOptions.addArguments("-headless");
                }
                firefoxOptions.setLogLevel(FirefoxDriverLogLevel.fromLevel(config.getWebdriverLogLevel()));
                firefoxOptions.addPreference("browser.helperApps.neverAsk.openFile", "application/octet-stream");
                return new FirefoxDriver(firefoxOptions);
            case "chrome":
                ChromeOptions chromeOptions = new ChromeOptions();
                chromeOptions.setPageLoadStrategy(PageLoadStrategy.NORMAL);
                if (config.isWebdriverHeadless()) {
                    chromeOptions.addArguments("--headless=new");
                }
                chromeOptions.setCapability(ChromeOptions.LOGGING_PREFS, logPrefs);
                chromeOptions.setLogLevel(ChromeDriverLogLevel.fromLevel(config.getWebdriverLogLevel()));

                // Chrome 111 workaround: https://github.com/SeleniumHQ/selenium/issues/11750
                chromeOptions.addArguments("--remote-allow-origins=*");

                Map<String, Object> chromePrefs = new HashMap<>();
                chromePrefs.put("download.prompt_for_download", false);
                chromePrefs.put("download.directory_upgrade", true);
                chromePrefs.put("safebrowsing.enabled", true);
                chromePrefs.put("profile.default_content_settings.popups", 0);
                chromePrefs.put("download.default_directory", System.getProperty("java.io.tmpdir"));
                chromeOptions.setExperimentalOption("prefs", chromePrefs);
                return new ChromeDriver(chromeOptions);
            case "safari":
                SafariOptions safariOptions = new SafariOptions();
                /*
                 * Safari does not support headless as of september 2020
                 * https://github.com/SeleniumHQ/selenium/issues/5985
                 * https://discussions.apple.com/thread/251837694
                 */
                //safariOptions.setHeadless(config.isDriverHeadless());
                safariOptions.setCapability("safari:diagnose", "true");
                return new SafariDriver(safariOptions);
        }

        throw new RuntimeException("Current supported browsers are: safari, firefox, chrome");
    }

}
