/*
 * Copyright 2009-2024 PrimeTek.
 *
 * https://www.primefaces.org/lts/licenses/
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.primefaces.selenium.spi;

import java.util.ArrayList;
import org.openqa.selenium.Dimension;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.WebDriverException;
import org.openqa.selenium.support.events.EventFiringDecorator;
import org.openqa.selenium.support.events.WebDriverListener;
import org.primefaces.selenium.PrimeSelenium;
import org.primefaces.selenium.internal.ConfigProvider;
import org.primefaces.selenium.internal.OnloadScriptsEventListener;
import org.primefaces.selenium.internal.ScrollElementIntoViewClickListener;

public class WebDriverProvider {

    private static final ThreadLocal<WebDriver> WEB_DRIVER = new ThreadLocal<>();

    private static final int CREATE_WEBDRIVER_RETRIES = 3;

    private WebDriverProvider() {

    }

    public static void set(WebDriver driver) {
        WEB_DRIVER.set(driver);
    }

    public static WebDriver get() {
        return get(false);
    }

    public static WebDriver get(boolean create) {
        WebDriver driver = WEB_DRIVER.get();
        if (driver == null && create) {
            WebDriverAdapter adapter = ConfigProvider.getInstance().getWebdriverAdapter();
            int fails = 0;

            do {
                /*
                 * Avoid issues like 2021-01-18T20:31:34.5805460Z org.openqa.selenium.WebDriverException: 2021-01-18T20:31:34.5810490Z
                 * java.net.ConnectException: Failed to connect to localhost/0:0:0:0:0:0:0:1:27231 which sometimes occur during Github Action jobs.
                 */
                try {
                    driver = adapter.createWebDriver();
                }
                catch (WebDriverException ex) {
                    fails++;
                    if (fails >= CREATE_WEBDRIVER_RETRIES) {
                        throw ex;
                    }
                }
            } while (driver == null);

            /*
             * Define window-size for headless-mode. Selenium WebDriver-default seems to be 800x600. This causes issues with modern themes (eg Saga) which use
             * more space for some components. (eg DatePicker-popup)
             */
            if (PrimeSelenium.isHeadless()) {
                driver.manage().window().setSize(new Dimension(1920, 1080));
            }
            else {
                driver.manage().window().setSize(new Dimension(1280, 1000));
            }

            ArrayList<WebDriverListener> listeners = new ArrayList<>();
            listeners.add(new OnloadScriptsEventListener());

            if (ConfigProvider.getInstance().getScrollElementIntoView() != null) {
                listeners.add(
                        new ScrollElementIntoViewClickListener(ConfigProvider.getInstance().getScrollElementIntoView()));
            }

            driver = new EventFiringDecorator(listeners.toArray(new WebDriverListener[listeners.size()])).decorate(driver);

            set(driver);
        }
        return driver;
    }
}
