/*
 * Copyright 2009-2024 PrimeTek.
 *
 * https://www.primefaces.org/lts/licenses/
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.primefaces.selenium.findby;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

import org.openqa.selenium.support.FindBy;

/**
 * Alternative {@link FindBy} annotation, which should only be used in page fragments. It searches by id, starting from the parent element, based on the id of
 * the parent concatenated with {@link #value()}. Sometimes, for example with <code>appendTo="..."</code>, the child element is moved to somewhere else in the
 * DOM. In this case you have to set {@link #searchFromRoot()} to <code>true</code>.
 * <p>
 * NOTE: If a sub-element under the parent does not have an ID but only a name you can use the name attribute to specify it and it will look for the name="xx"
 * instead of ID.
 * </p>
 */
@Retention(RetentionPolicy.RUNTIME)
@Target(ElementType.FIELD)
public @interface FindByParentPartialId {

    /**
     * ID of the component under the parent to find.
     *
     * @return the ID
     */
    String value();

    /**
     * Name attribute to find instead of using the ID of the component.
     *
     * @return the name attribute or default "".
     */
    String name() default "";

    boolean searchFromRoot() default false;
}
