/*
 * Copyright 2009-2024 PrimeTek.
 *
 * https://www.primefaces.org/lts/licenses/
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.primefaces.selenium;

import java.io.File;

import org.openqa.selenium.JavascriptExecutor;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.WebElement;
import org.openqa.selenium.support.ui.ExpectedCondition;
import org.openqa.selenium.support.ui.ExpectedConditions;

public final class PrimeExpectedConditions {

    private PrimeExpectedConditions() {
        super();
    }

    public static ExpectedCondition<Boolean> script(String script) {
        return driver -> (Boolean) ((JavascriptExecutor) driver).executeScript(script);
    }

    public static ExpectedCondition<Boolean> documentLoaded() {
        return script("return document.readyState === 'complete'");
    }

    public static ExpectedCondition<Boolean> notNavigating() {
        return script("return (!window.pfselenium || pfselenium.navigating === false);");
    }

    public static ExpectedCondition<Boolean> notSubmitting() {
        return script("return (!window.pfselenium || pfselenium.submitting === false);");
    }

    public static ExpectedCondition<Boolean> animationNotActive() {
        return script("return ((!window.jQuery || jQuery.active == 0) && (!window.PrimeFaces || PrimeFaces.animationActive === false));");
    }

    public static ExpectedCondition<Boolean> ajaxQueueEmpty() {
        return script("return (!window.PrimeFaces || PrimeFaces.ajax.Queue.isEmpty());");
    }

    public static ExpectedCondition<Boolean> elementToBeClickable(WebElement element) {
        return ExpectedConditions.and(ExpectedConditions.elementToBeClickable(element));
    }

    public static ExpectedCondition<Boolean> visibleAndAnimationComplete(WebElement element) {
        return ExpectedConditions.and(
                    documentLoaded(),
                    animationNotActive(),
                    ajaxQueueEmpty(),
                    ExpectedConditions.visibilityOf(element));
    }

    public static ExpectedCondition<Boolean> invisibleAndAnimationComplete(WebElement element) {
        return ExpectedConditions.and(
                    documentLoaded(),
                    animationNotActive(),
                    ajaxQueueEmpty(),
                    ExpectedConditions.invisibilityOf(element));
    }

    public static ExpectedCondition<Boolean> visibleInViewport(WebElement element) {
        return new ExpectedCondition<Boolean>() {
            @Override
            public Boolean apply(WebDriver webDriver) {
                return (Boolean) PrimeSelenium.isVisibleInViewport(element);
            }

            @Override
            public String toString() {
                return "is " + element + " visible in viewport";
            }
        };
    }

    public static ExpectedCondition<Boolean> fileExists(String path) {
        return new ExpectedCondition<Boolean>() {
            @Override
            public Boolean apply(WebDriver driver) {
                File f = new File(path);
                return f.exists() && f.isFile();
            }

            @Override
            public String toString() {
                return String.format("File '%s' to be present within the time specified", path);
            }
        };
    }

}
