/*
 * Copyright 2009-2024 PrimeTek.
 *
 * https://www.primefaces.org/lts/licenses/
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.primefaces.selenium.component.html;

import org.openqa.selenium.By;
import org.openqa.selenium.WebElement;
import org.primefaces.selenium.component.base.AbstractInputComponent;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import org.primefaces.selenium.PrimeSelenium;

/**
 * Component wrapper for the Faces {@code h:selectOneMenu}.
 */
public abstract class SelectOneMenu extends AbstractInputComponent {

    public void deselect(String label) {
        if (!isSelected(label) || !isEnabled()) {
            return;
        }

        WebElement option = getOptions().stream()
                .filter(e -> Objects.equals(e.getAttribute("innerHTML"), label) && e.isSelected())
                .findFirst()
                .orElse(null);
        if (option != null) {
            if (isAjaxified("onchange")) {
                option = PrimeSelenium.guardAjax(option);
            }
            option.click();
        }
    }

    public void select(String label) {
        if (isSelected(label) || !isEnabled()) {
            return;
        }

        WebElement option = getOptions().stream()
                .filter(e -> Objects.equals(e.getAttribute("innerHTML"), label) && !e.isSelected())
                .findFirst()
                .orElse(null);
        if (option != null) {
            if (isAjaxified("onchange")) {
                option = PrimeSelenium.guardAjax(option);
            }
            option.click();
        }
    }

    public String getSelectedLabel() {
        WebElement option = getOptions().stream()
                .filter(e -> e.isSelected())
                .findFirst()
                .orElse(null);
        return option == null ? null : option.getAttribute("innerHTML");
    }

    public boolean isSelected(String label) {
        boolean result = false;
        try {
            result = getSelectedLabel().equalsIgnoreCase(label);
        }
        catch (Exception e) {
            // do nothing
        }
        return result;
    }

    /**
     * All labels.
     * @return
     */
    public List<String> getLabels() {
        return getOptions().stream()
                .map(e -> e.getAttribute("innerHTML"))
                .collect(Collectors.toList());
    }

    public List<WebElement> getOptions() {
        return getRoot().findElements(By.tagName("option"));
    }

    public void select(int index) {
        if (isSelected(index)) {
            return;
        }

        select(getLabel(index));
    }

    public void deselect(int index) {
        if (!isSelected(index)) {
            return;
        }

        deselect(getLabel(index));
    }

    public boolean isSelected(int index) {
        return getLabel(index).equals(getSelectedLabel());
    }

    public String getLabel(int index) {
        return getLabels().get(index);
    }

    @Override
    public WebElement getInput() {
        return getRoot();
    }
}
