/*
 * Copyright 2009-2024 PrimeTek.
 *
 * https://www.primefaces.org/lts/licenses/
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.primefaces.selenium.component.base;

import org.json.JSONObject;
import org.openqa.selenium.WebElement;
import org.primefaces.selenium.AbstractPrimePageFragment;
import org.primefaces.selenium.PrimeSelenium;

public abstract class AbstractComponent extends AbstractPrimePageFragment {

    /**
     * CSP script using jQuery events to see if an event is AJAXified.
     */
    private static final String CSP_SCRIPT = "return PrimeFaces.csp.hasRegisteredAjaxifiedEvent('%s', '%s');";

    /**
     * Gets the widget by component id JS function.
     *
     * @return the JS script
     */
    public String getWidgetByIdScript() {
        return ComponentUtils.getWidgetByIdScript(getId());
    }

    /**
     * Gets the current widget's configuration e.g. widget.cfg as a String.
     *
     * @return the String representation of the widget configuration
     */
    public String getWidgetConfigurationAsString() {
        return ComponentUtils.getWidgetConfiguration(getRoot());
    }

    /**
     * Gets the current widget's configuration e.g. widget.cfg as a JSON object.
     *
     * @return the {@link JSONObject} representing the config, useful for assertions
     */
    public JSONObject getWidgetConfiguration() {
        String cfg = getWidgetConfigurationAsString();
        if (cfg == null || cfg.length() == 0) {
            return null;
        }
        return new JSONObject(cfg);
    }

    /**
     * Is the event for the root-element AJAXified?
     *
     * @param event Event with the `on` prefix, such as `onclick` or `onblur`.
     * @return true if using AJAX false if not
     */
    protected boolean isAjaxified(String event) {
        return isAjaxified(getRoot(), event);
    }

    /**
     * Is the event AJAXified?
     *
     * @param element Element for which to do the check. (May be a child element of a complex component.) If no element is passed it defaults to getRoot().
     * @param event Event with the `on` prefix, such as `onclick` or `onblur`.
     * @return true is using AJAX false it not
     */
    protected boolean isAjaxified(WebElement element, String event) {
        if (element == null) {
            element = getRoot();
        }

        // first check normal path if component is AJAXified
        boolean isAjaxScript = ComponentUtils.isAjaxScript(element.getAttribute(event));
        if (isAjaxScript) {
            return true;
        }

        // now check for CSP events
        String id = element.getAttribute("id");
        String cspScript = String.format(CSP_SCRIPT, id, event);
        return PrimeSelenium.executeScript(cspScript);
    }

    /**
     * Destroy the widget.
     */
    public void destroy() {
        PrimeSelenium.executeScript(getWidgetByIdScript() + ".destroy();");
    }
}
