/*
 * Copyright 2009-2024 PrimeTek.
 *
 * https://www.primefaces.org/lts/licenses/
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.primefaces.selenium.component;

import java.util.List;

import org.openqa.selenium.WebElement;
import org.openqa.selenium.support.FindBy;
import org.primefaces.selenium.PrimeExpectedConditions;
import org.primefaces.selenium.PrimeSelenium;
import org.primefaces.selenium.component.base.AbstractInputComponent;
import org.primefaces.selenium.component.base.ComponentUtils;
import org.primefaces.selenium.findby.FindByParentPartialId;

/**
 * Component wrapper for the PrimeFaces {@code selectCheckboxMenu }.
 */
public abstract class SelectCheckboxMenu extends AbstractInputComponent {

    @FindByParentPartialId("_input")
    private WebElement input;

    @FindByParentPartialId(value = "_panel", searchFromRoot = true)
    private WebElement panel;

    @FindByParentPartialId(value = "_filter", searchFromRoot = true)
    private WebElement filterInput;

    @FindBy(css = "input[type='checkbox']")
    private List<WebElement> checkboxes;

    @FindBy(css = "input[type='checkbox']:checked")
    private List<WebElement> selectedCheckboxes;

    public WebElement getFilterInput() {
        return filterInput;
    }

    public WebElement getPanel() {
        return panel;
    }

    public List<WebElement> getCheckboxes() {
        return checkboxes;
    }

    public List<WebElement> getSelectedCheckboxes() {
        return selectedCheckboxes;
    }

    /**
     * Is the input using AJAX "itemSelect" event?
     *
     * @return true if using AJAX for itemSelect
     */
    public boolean isItemSelectAjaxified() {
        return ComponentUtils.hasAjaxBehavior(getRoot(), "itemSelect");
    }

    /**
     * Is the input using AJAX "itemUnselect" event?
     *
     * @return true if using AJAX for itemUnselect
     */
    public boolean isItemUnselectAjaxified() {
        return ComponentUtils.hasAjaxBehavior(getRoot(), "itemUnselect");
    }

    /**
     * Bring up the overlay panel if its not showing or hide it if it is showing.
     */
    public void togglPanel() {
        PrimeSelenium.executeScript(getWidgetByIdScript() + ".togglePanel();");
    }

    /**
     * Shows the SelectOneMenu panel.
     */
    public void show() {
        WebElement panel = getPanel();
        if (isEnabled() && !panel.isDisplayed()) {
            PrimeSelenium.executeScript(getWidgetByIdScript() + ".show();");
            PrimeSelenium.waitGui().until(PrimeExpectedConditions.visibleAndAnimationComplete(panel));
        }
    }

    /**
     * Hides the SelectOneMenu panel.
     */
    public void hide() {
        WebElement panel = getPanel();
        if (isEnabled() && panel.isDisplayed()) {
            PrimeSelenium.executeScript(getWidgetByIdScript() + ".hide();");
            PrimeSelenium.waitGui().until(PrimeExpectedConditions.invisibleAndAnimationComplete(panel));
        }
    }

    /**
     * Selects all available options.
     */
    public void checkAll() {
        if (isEnabled()) {
            PrimeSelenium.executeScript(getWidgetByIdScript() + ".checkAll();");
        }
    }

    /**
     * Unselects all available options.
     */
    public void uncheckAll() {
        if (isEnabled()) {
            PrimeSelenium.executeScript(getWidgetByIdScript() + ".uncheckAll();");
        }
    }

    /**
     * Resets the input.
     */
    public void resetValue() {
        if (isEnabled()) {
            PrimeSelenium.executeScript(getWidgetByIdScript() + ".resetValue();");
        }
    }

    /**
     * Checks the checkbox option with the given value.
     *
     * @param value the value to check
     */
    public void selectValue(String value) {
        if (isEnabled()) {
            PrimeSelenium.executeScript(getWidgetByIdScript() + ".selectValue('" + value + "');");
        }
    }

}
