/*
 * Copyright 2009-2024 PrimeTek.
 *
 * https://www.primefaces.org/lts/licenses/
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.primefaces.selenium.component;

import org.openqa.selenium.By;
import org.openqa.selenium.WebElement;
import org.primefaces.selenium.PrimeSelenium;
import org.primefaces.selenium.component.base.AbstractInputComponent;
import org.primefaces.selenium.component.base.ComponentUtils;
import org.primefaces.selenium.findby.FindByParentPartialId;

/**
 * Component wrapper for the PrimeFaces {@code p:rating}.
 */
public abstract class Rating extends AbstractInputComponent {

    @FindByParentPartialId("_input")
    private WebElement input;

    @Override
    public WebElement getInput() {
        return input;
    }

    /**
     * Is Cancel event AJAX enabled?
     *
     * @return true if AJAX enabled false if not
     */
    public boolean isCancelAjaxified() {
        return ComponentUtils.hasBehavior(this, "cancel") || isOnchangeAjaxified();
    }

    /**
     * Is Rating event AJAX enabled?
     *
     * @return true if AJAX enabled false if not
     */
    public boolean isRatingAjaxified() {
        return ComponentUtils.hasBehavior(this, "rate") || isOnchangeAjaxified();
    }

    /**
     * Gets the cancel icon if available.
     *
     * @return the cancel icon
     */
    public WebElement getCancelIcon() {
        return findElement(By.className("ui-rating-cancel"));
    }

    /**
     * Resets the rating so that no stars are selected using the cancel icon.
     */
    public void cancel() {
        WebElement cancelIcon = getCancelIcon();
        if (isCancelAjaxified() || isRatingAjaxified()) {
            PrimeSelenium.guardAjax(cancelIcon).click();
        }
        else {
            cancelIcon.click();
        }

    }

    /**
     * Finds the current rating, i.e. the number of stars selected.
     *
     * @return The current rating value.
     */
    public Number getValue() {
        return PrimeSelenium.executeScript("return " + getWidgetByIdScript() + ".getValue();");
    }

    /**
     * Sets the rating to the given value.
     *
     * @param value New rating value to set (number of starts selected).
     */
    public void setValue(Number value) {
        PrimeSelenium.executeScript(isRatingAjaxified(), getWidgetByIdScript() + ".setValue(" + value + ");");
    }

    /**
     * Resets the rating so that no stars are selected.
     */
    public void reset() {
        PrimeSelenium.executeScript(isCancelAjaxified(), getWidgetByIdScript() + ".reset();");
    }

    /**
     * Is this rating disabled?
     *
     * @return true if disabled
     */
    public boolean isDisabled() {
        return PrimeSelenium.executeScript("return " + getWidgetByIdScript() + ".isDisabled();");
    }

    /**
     * Is this rating readonly?
     *
     * @return true if readonly
     */
    public boolean isReadOnly() {
        return PrimeSelenium.executeScript("return " + getWidgetByIdScript() + ".isReadOnly();");
    }

    @Override
    public boolean isEnabled() {
        return !isDisabled();
    }
}
