/*
 * Copyright 2009-2024 PrimeTek.
 *
 * https://www.primefaces.org/lts/licenses/
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.primefaces.selenium.component;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import org.openqa.selenium.By;
import org.openqa.selenium.NoSuchElementException;
import org.openqa.selenium.WebElement;
import org.primefaces.selenium.component.base.AbstractComponent;
import org.primefaces.selenium.component.model.Msg;
import org.primefaces.selenium.component.model.Severity;

/**
 * Component wrapper for the PrimeFaces {@code p:messages}.
 */
public abstract class Messages extends AbstractComponent {

    public boolean isEmpty() {
        if (!isDisplayed()) {
            return true;
        }
        return getAllMessages().isEmpty();
    }

    public List<Msg> getAllMessages() {
        List<Msg> result = new ArrayList<>();
        if (!isDisplayed()) {
            return result;
        }

        List<WebElement> messagesSeverities = findElements(By.tagName("div"));
        for (WebElement messageSeverity : messagesSeverities) {

            Severity severity = Severity.toSeverity(messageSeverity.getAttribute("class"));

            for (WebElement message : messageSeverity.findElements(By.cssSelector("li"))) {
                Msg msg = new Msg();
                msg.setSeverity(severity);
                msg.setSummary(message.findElement(By.className("ui-messages-" + Severity.toName(severity) + "-summary")).getText());
                try {
                    msg.setDetail(message.findElement(By.className("ui-messages-" + Severity.toName(severity) + "-detail")).getText());
                }
                catch (NoSuchElementException e) {
                    // ignore, it's optional
                }

                result.add(msg);
            }
        }

        return result;
    }

    public Msg getMessage(int index) {
        List<Msg> allMessages = getAllMessages();

        if (allMessages.size() > index) {
            return allMessages.get(index);
        }

        return null;
    }

    public List<Msg> getMessagesBySeverity(Severity severity) {
        return getAllMessages().stream()
                    .filter(message -> severity.equals(message.getSeverity()))
                    .collect(Collectors.toList());
    }

    public List<String> getAllSummaries() {
        return getAllMessages().stream()
                    .map(Msg::getSummary)
                    .collect(Collectors.toList());
    }
}
