/*
 * Copyright 2009-2024 PrimeTek.
 *
 * https://www.primefaces.org/lts/licenses/
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.primefaces.selenium.component;

import java.io.File;
import java.io.Serializable;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;
import org.openqa.selenium.By;
import org.openqa.selenium.NoSuchElementException;

import org.openqa.selenium.WebElement;
import org.primefaces.selenium.PrimeExpectedConditions;
import org.primefaces.selenium.PrimeSelenium;
import org.primefaces.selenium.component.base.AbstractInputComponent;
import org.primefaces.selenium.internal.ConfigProvider;
import org.primefaces.selenium.internal.Guard;

/**
 * Component wrapper for the PrimeFaces {@code p:fileUpload}.
 */
public abstract class FileUpload extends AbstractInputComponent {

    @Override
    public WebElement getInput() {
        // in case of mode=simple skinSimple=false the input element is this element
        boolean isInputFile = "input".equals(getTagName()) && "file".equals(getAttribute("type"));
        return isInputFile ? this : findElement(By.id(getId() + "_input"));
    }

    /**
     * Returns the input´s value.
     * @return the file name
     */
    public String getValue() {
        return getInput().getAttribute("value");
    }

    /**
     * Sets the input's value.
     * This should be the files absolute path.
     * @param value the file name to set
     */
    public void setValue(Serializable value) {
        setValue(value, true);
    }

    /**
     * Sets the input's value.
     * This should be the files absolute path.
     * @param value the file name to
     * @param useGuard use guard to wait until the upload finished
     */
    public void setValue(Serializable value, boolean useGuard) {
        Runnable runnable = () -> {
            if (getInput() != this && !PrimeSelenium.isChrome()) {
                // input file cannot be cleared if skinSimple=false or in Chrome
                getInput().clear();
            }
            // ComponentUtils.sendKeys will break tests in Chrome
            getInput().sendKeys(value.toString());

            PrimeSelenium.wait(200);
        };

        if (isAutoUpload()) {
            if (useGuard) {
                if (isAdvancedMode()) {
                    Runnable guarded = Guard.custom(
                            runnable,
                            200,
                            ConfigProvider.getInstance().getTimeoutFileUpload(),
                            PrimeExpectedConditions.script("return " + getWidgetByIdScript() + ".files.length === 0;"));

                    guarded.run();
                }
                else {
                    PrimeSelenium.guardAjax(runnable).run();
                }
            }
            else {
                runnable.run();
            }
        }
        else {
            runnable.run();
        }
    }

    /**
     * Sets the input's value from given files.
     * @param values the file name(s) to set
     */
    public void setValue(File... values) {
        setValue(true, values);
    }

    /**
     * Sets the input's value from given files.
     * @param values the file name(s) to set
     * @param useGuard use guard to wait until the upload finished
     */
    public void setValue(boolean useGuard, File... values) {
        // several references in the WEB state that (at least) Firefox and Chrome accept
        // multiple filenames separated by a new line character
        String paths = Arrays.stream(values)
                .map(f -> f.getAbsolutePath())
                .collect(Collectors.joining("\n"));
        setValue(paths, useGuard);
    }

    /**
     * Gets the Upload button of the widget.
     * The button is only rendered in advanced mode.
     * @return the widget's upload button
     */
    public WebElement getAdvancedUploadButton() {
        WebElement element = findElement(By.cssSelector(".ui-fileupload-buttonbar button.ui-fileupload-upload"));

        WebElement guarded = Guard.custom(
            element,
            200,
            ConfigProvider.getInstance().getTimeoutFileUpload(),
            PrimeExpectedConditions.script("return " + getWidgetByIdScript() + ".files.length === 0;"));

        return guarded;
    }

    /**
     * Gets the Cancel button of the widget.
     * The button is only rendered in advanced mode.
     * @return the widget's cancel button
     */
    public WebElement getAdvancedCancelButton() {
        return findElement(By.cssSelector(".ui-fileupload-buttonbar button.ui-fileupload-cancel"));
    }

    /**
     * Gets the file Cancel button of the widget.
     * The button is only rendered in advanced mode.
     * @param fileName the file name for which to return the cancel button
     * @return the widget's cancel button
     */
    public WebElement getAdvancedCancelButton(String fileName) {
        for (WebElement row : findElements(By.cssSelector(".ui-fileupload-files .ui-fileupload-row"))) {
            WebElement fn = row.findElement(By.cssSelector(".ui-fileupload-filename"));
            if (fn.getText().contains(fileName)) {
                return row.findElement(By.cssSelector("button.ui-fileupload-cancel"));
            }
        }
        throw new NoSuchElementException("cancel button for " + fileName + " not found");
    }

    /**
     * Gets the value displayed by the widget.
     *
     * @return the widget's value
     */
    public String getWidgetValue() {
        return findElement(By.className("ui-fileupload-filename")).getText();
    }

    /**
     * Gets the values displayed by the widget.
     *
     * @return the widget's values
     */
    public List<String> getWidgetValues() {
        return findElements(By.className("ui-fileupload-filename")).stream()
                .map(e -> e.getText()).collect(Collectors.toList());
    }

    /**
     * Gets the values displayed by the widget.
     *
     * @return the widget's error messages
     */
    public List<String> getWidgetErrorMessages() {
        return findElements(By.className("ui-messages-error-summary")).stream()
                .map(e -> e.getText()).collect(Collectors.toList());
    }

    public boolean isAdvancedMode() {
        return "advanced".equals(getWidgetConfiguration().getString("mode"));
    }

    public boolean isAutoUpload() {
        return getWidgetConfiguration().has("auto") ? getWidgetConfiguration().getBoolean("auto") : false;
    }
}
