/*
 * Copyright 2009-2024 PrimeTek.
 *
 * https://www.primefaces.org/lts/licenses/
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.primefaces.selenium.component;

import java.util.List;

import org.openqa.selenium.By;
import org.openqa.selenium.WebElement;
import org.primefaces.selenium.PrimeExpectedConditions;
import org.primefaces.selenium.PrimeSelenium;
import org.primefaces.selenium.component.base.AbstractComponent;

/**
 * Component wrapper for the PrimeFaces {@code p:blockUI}.
 */
public abstract class BlockUI extends AbstractComponent {

    /**
     * Finds all Block UI overlay elements by class name.
     *
     * @return a List<WebElement> containing at least one element
     */
    public List<WebElement> getOverlays() {
        return getWebDriver().findElements(By.className("ui-blockui"));
    }

    /**
     * Finds all Block UI content elements by class name.
     *
     * @return a List<WebElement> containing at least one element
     */
    public List<WebElement> getContents() {
        return getWebDriver().findElements(By.className("ui-blockui-content"));
    }

    /**
     * Is the blocker currently blocking.
     *
     * @return true if blocking false if not
     */
    public boolean isBlocking() {
        return PrimeSelenium.executeScript("return " + getWidgetByIdScript() + ".isBlocking();");
    }

    /**
     * Does the block have content.
     *
     * @return true if has content, false if not
     */
    public boolean hasContent() {
        return PrimeSelenium.executeScript("return " + getWidgetByIdScript() + ".hasContent();");
    }

    /**
     * Shows the blocker.
     */
    public void show() {
        PrimeSelenium.executeScript(getWidgetByIdScript() + ".show();");
        List<WebElement> overlays = getOverlays();
        if (overlays.size() > 0) {
            PrimeSelenium.waitGui().until(PrimeExpectedConditions.visibleAndAnimationComplete(overlays.get(0)));
        }
    }

    /**
     * Hides the blocker.
     */
    public void hide() {
        PrimeSelenium.executeScript(getWidgetByIdScript() + ".hide();");
        List<WebElement> overlays = getOverlays();
        if (overlays.size() > 0) {
            PrimeSelenium.waitGui().until(PrimeExpectedConditions.invisibleAndAnimationComplete(overlays.get(0)));
        }
    }

}
