/*
 * Copyright 2009-2024 PrimeTek.
 *
 * https://www.primefaces.org/lts/licenses/
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.primefaces.selenium.component;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import org.openqa.selenium.WebElement;
import org.openqa.selenium.support.FindBy;
import org.primefaces.selenium.PrimeSelenium;
import org.primefaces.selenium.component.base.AbstractComponent;
import org.primefaces.selenium.component.base.ComponentUtils;
import org.primefaces.selenium.component.model.Tab;

/**
 * Component wrapper for the PrimeFaces {@code p:accordionPanel}.
 */
public abstract class AccordionPanel extends AbstractComponent {

    @FindBy(css = ".ui-accordion-header")
    private List<WebElement> headers;

    @FindBy(css = ".ui-accordion-content")
    private List<WebElement> contents;

    private List<Tab> tabs = null;

    public List<WebElement> getHeaders() {
        return headers;
    }

    public List<WebElement> getContents() {
        return contents;
    }

    /**
     * Gets the accordion tabs.
     *
     * @return the list of tabs
     */
    public List<Tab> getTabs() {
        if (tabs == null) {
            List<Tab> tabs = new ArrayList<>();

            AtomicInteger cnt = new AtomicInteger(0);
            headers.forEach(headerElt -> {
                String title = headerElt.getText();
                int index = cnt.getAndIncrement();
                WebElement content = getContents().get(index);

                tabs.add(new Tab(title, index, headerElt, content));
            });

            this.tabs = tabs;
        }

        return tabs;
    }

    /**
     * Toggle the tab denoted by the specified index.
     *
     * @param index the index of the tab to toggle
     */
    public void toggleTab(int index) {
        if (ComponentUtils.hasAjaxBehavior(getRoot(), "tabChange")) {
            PrimeSelenium.guardAjax(getHeaders().get(index)).click();
        }
        else {
            getHeaders().get(index).click();
        }
    }

    /**
     * Expands the tab denoted by the specified index.
     *
     * @param index the index of the tab to expand
     */
    public void expandTab(int index) {
        WebElement tab = getHeaders().get(index);
        if (tab.getAttribute("aria-expanded").equalsIgnoreCase("false")) {
            toggleTab(index);
        }
    }

    /**
     * Collapse the tab denoted by the specified index.
     *
     * @param index the index of the tab to collapse
     */
    public void collapseTab(int index) {
        WebElement tab = getHeaders().get(index);
        if (tab.getAttribute("aria-expanded").equalsIgnoreCase("true")) {
            toggleTab(index);
        }
    }

    /**
     * Provides the selected {@link AccordionPanel} tab(s).
     *
     * @return the selected tab(s)
     */
    public List<Tab> getSelectedTabs() {
        return getTabs().stream()
                    .filter(tab -> tab.getHeader().getAttribute("class").contains("ui-state-active"))
                    .collect(Collectors.toList());
    }
}
