/*
 * Copyright 2009-2024 PrimeTek.
 *
 * https://www.primefaces.org/lts/licenses/
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.primefaces.selenium.component.base;

import org.openqa.selenium.WebElement;
import org.openqa.selenium.support.ui.ExpectedConditions;
import org.primefaces.selenium.PrimeExpectedConditions;
import org.primefaces.selenium.PrimeSelenium;
import org.primefaces.selenium.findby.FindByParentPartialId;

/**
 * Base class for boolean toggle components.
 */
public abstract class AbstractToggleComponent extends AbstractInputComponent {

    @FindByParentPartialId("_input")
    private WebElement input;

    @Override
    public WebElement getInput() {
        return input;
    }

    @Override
    public boolean isSelected() {
        return getValue();
    }

    @Override
    public void click() {
        PrimeSelenium.waitGui().until(PrimeExpectedConditions.visibleAndAnimationComplete(getRoot()));
        PrimeSelenium.waitGui().until(ExpectedConditions.elementToBeClickable(getRoot()));

        if (isOnChangeAjaxified()) {
            PrimeSelenium.guardAjax(getRoot()).click();
        }
        else {
            getRoot().click();
        }
    }

    /**
     * Is this toggle component AJAX enabled?
     *
     * @return true if AJAX enabled false if not
     */
    public boolean isOnChangeAjaxified() {
        return isAjaxified(getInput(), "onchange") || ComponentUtils.hasAjaxBehavior(getRoot(), "change");
    }

    /**
     * Set the value of the the toggle component.
     *
     * @param value true for checked, false for unchecked
     */
    public void setValue(boolean value) {
        if (getValue() != value) {
            click();
        }
    }

    /**
     * Gets the value of the toggle component.
     *
     * @return true for checked, false for unchecked
     */
    public boolean getValue() {
        return getInput().getAttribute("checked") != null;
    }

    /**
     * Turns this switch in case it is off, or turns of off in case it is on.
     */
    public void toggle() {
        PrimeSelenium.executeScript(isOnChangeAjaxified(), getWidgetByIdScript() + ".toggle();");
    }

    /**
     * Turns this switch on if it is not already turned on.
     */
    public void check() {
        PrimeSelenium.executeScript(isOnChangeAjaxified(), getWidgetByIdScript() + ".check();");
    }

    /**
     * Turns this switch off if it is not already turned of.
     */
    public void uncheck() {
        PrimeSelenium.executeScript(isOnChangeAjaxified(), getWidgetByIdScript() + ".uncheck();");
    }

}
