/*
 * Copyright 2009-2024 PrimeTek.
 *
 * https://www.primefaces.org/lts/licenses/
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.primefaces.selenium.component.base;

import org.openqa.selenium.By;
import org.openqa.selenium.Keys;
import org.openqa.selenium.WebElement;
import org.primefaces.selenium.PrimeSelenium;

public abstract class AbstractInputComponent extends AbstractComponent {

    /**
     * The input element reference.
     *
     * @return the {@link WebElement} representing the input.
     */
    public WebElement getInput() {
        return getRoot();
    }

    /**
     * Is this SelectOneMenu enabled?
     *
     * @return true if enabled, false if not
     */
    @Override
    public boolean isEnabled() {
        return getInput().isEnabled() && !PrimeSelenium.hasCssClass(this, "ui-state-disabled");
    }

    /**
     * Is the input using AJAX "change" or "valueChange" event?
     *
     * @return true if using AJAX for onchange, change or valueChange
     */
    public boolean isOnchangeAjaxified() {
        return isAjaxified(getInput(), "onchange") || ComponentUtils.hasAjaxBehavior(getRoot(), "change");
    }

    /**
     * The HTML label assigned to this input.
     *
     * @return the {@link WebElement} representing the label.
     */
    public WebElement getAssignedLabel() {
        return getWebDriver().findElement(By.cssSelector("label[for='" + getInput().getAttribute("id") + "']"));
    }

    /**
     * The HTML label text assigned to this input.
     *
     * @return the value of the label text
     */
    public String getAssignedLabelText() {
        return getAssignedLabel().getText();
    }

    /**
     * Copy the current value in the Input to the clipboard.
     *
     * @return the value copied to the clipboard
     */
    public String copyToClipboard() {
        WebElement input = getInput();
        Keys command = PrimeSelenium.isMacOs() ? Keys.COMMAND : Keys.CONTROL;
        input.sendKeys(Keys.chord(command, "a")); // select everything
        input.sendKeys(Keys.chord(command, "c")); // copy
        return input.getAttribute("value");
    }

    /**
     * Paste the current value of the clipboard to the Input.
     *
     * @return the value pasted into the input
     */
    public String pasteFromClipboard() {
        WebElement input = getInput();
        Keys command = PrimeSelenium.isMacOs() ? Keys.COMMAND : Keys.CONTROL;
        input.sendKeys(Keys.chord(command, "a")); // select everything
        input.sendKeys(Keys.chord(command, "v")); // paste
        return input.getAttribute("value");
    }

    /**
     * Selects all text in the input component.
     */
    public void selectAllText() {
        Keys command = PrimeSelenium.isMacOs() ? Keys.COMMAND : Keys.CONTROL;
        getInput().sendKeys(Keys.chord(command, "a"));
    }

    /**
     * Clears the input and guards AJAX for "clear" event.
     */
    @Override
    public void clear() {
        PrimeSelenium.clearInput(getInput(), false);
    }

    /**
     * Enables the input/
     */
    public void enable() {
        PrimeSelenium.executeScript(getWidgetByIdScript() + ".enable();");
    }

    /**
     * Disables the input.
     */
    public void disable() {
        PrimeSelenium.executeScript(getWidgetByIdScript() + ".disable();");
    }

}
