/*
 * Copyright 2009-2024 PrimeTek.
 *
 * https://www.primefaces.org/lts/licenses/
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.primefaces.selenium.component;

import org.openqa.selenium.By;
import org.openqa.selenium.WebElement;
import org.primefaces.selenium.PrimeSelenium;
import org.primefaces.selenium.component.base.AbstractComponent;
import org.primefaces.selenium.component.base.ComponentUtils;

/**
 * Component wrapper for the PrimeFaces {@code p:timeline}.
 */
public abstract class Timeline extends AbstractComponent {

    /**
     * Selects either an event or any other element in the timeline by its CSS class.
     *
     * @param cssClass the CSS class to select
     */
    public void select(String cssClass) {
        WebElement element = findElement(By.className(cssClass));
        if (isSelectAjaxified()) {
            PrimeSelenium.guardAjax(element).click();
        }
        else {
            element.click();
        }
    }

    /**
     * Is the timeline using AJAX "rangechanged" event?
     *
     * @return true if using AJAX for rangechanged
     */
    public boolean isRangeChangedAjaxified() {
        return ComponentUtils.hasAjaxBehavior(getRoot(), "rangechanged");
    }

    /**
     * Is the timeline using AJAX "select" event?
     *
     * @return true if using AJAX for select
     */
    public boolean isSelectAjaxified() {
        return ComponentUtils.hasAjaxBehavior(getRoot(), "select");
    }

    /**
     * Force render the timeline component.
     */
    public void update() {
        PrimeSelenium.executeScript(getWidgetByIdScript() + ".renderTimeline();");
    }

    /**
     * Gets number of events (items in the timeline).
     *
     * @return The number of event in the timeline.
     */
    public long getNumberOfEvents() {
        return PrimeSelenium.executeScript("return " + getWidgetByIdScript() + ".getNumberOfEvents();");
    }

    /**
     * Zoom the timeline in by the zoom factor.
     *
     * @param zoomFactor a number between 0 and 1
     */
    public void zoom(double zoomFactor) {
        PrimeSelenium.executeScript(isRangeChangedAjaxified(), getWidgetByIdScript() + ".zoom(" + zoomFactor + ");");
    }

    /**
     * Move the timeline left or right by a factor
     *
     * @param moveFactor a number between 0 and 1
     */
    public void move(double moveFactor) {
        PrimeSelenium.executeScript(isRangeChangedAjaxified(), getWidgetByIdScript() + ".move(" + moveFactor + ");");
    }
}
