/*
 * Copyright 2009-2024 PrimeTek.
 *
 * https://www.primefaces.org/lts/licenses/
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.primefaces.selenium.component;

import org.primefaces.selenium.PrimeExpectedConditions;
import org.primefaces.selenium.PrimeSelenium;
import org.primefaces.selenium.component.base.AbstractComponent;

/**
 * Component wrapper for the PrimeFaces {@code p:overlayPanel}.
 */
public abstract class OverlayPanel extends AbstractComponent {

    /**
     * Is the overlay currently visible.
     *
     * @return true if visible false if not
     */
    public boolean isVisible() {
        return PrimeSelenium.executeScript("return " + getWidgetByIdScript() + ".isVisible();");
    }

    /**
     * Is the overlay currently visible.
     *
     * @return true if visible false if not
     */
    public int getShowDelay() {
        return getWidgetConfiguration().getInt("showDelay");
    }

    /**
     * Toggle the overlay visibility.
     */
    public void toggle() {
        PrimeSelenium.executeScript(getWidgetByIdScript() + ".toggle();");
    }

    /**
     * Makes this overlay modal.
     */
    public void enableModality() {
        PrimeSelenium.executeScript(getWidgetByIdScript() + ".enableModality();");
    }

    /**
     * Makes this overlay non-modal.
     */
    public void disableModality() {
        PrimeSelenium.executeScript(getWidgetByIdScript() + ".disableModality();");
    }

    /**
     * Shows the overlay.
     */
    public void show() {
        if (isEnabled() && !isDisplayed()) {
            PrimeSelenium.executeScript(getWidgetByIdScript() + ".show();");
            waitForDisplay();
        }
    }

    /**
     * Hides the overlay.
     */
    public void hide() {
        if (isEnabled() && isDisplayed()) {
            PrimeSelenium.executeScript(getWidgetByIdScript() + ".hide();");
            PrimeSelenium.waitGui().until(PrimeExpectedConditions.invisibleAndAnimationComplete(this));
        }
    }

    /**
     * Wait for the client side setTimeout when displaying overlay.
     */
    public void waitForDisplay() {
        PrimeSelenium.wait(getShowDelay());
        PrimeSelenium.waitGui().until(PrimeExpectedConditions.visibleAndAnimationComplete(this));
    }

}
