/*
 * Copyright 2009-2024 PrimeTek.
 *
 * https://www.primefaces.org/lts/licenses/
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.primefaces.selenium.component;

import org.openqa.selenium.By;
import org.openqa.selenium.NoSuchElementException;
import org.openqa.selenium.WebElement;
import org.openqa.selenium.interactions.Actions;
import org.primefaces.selenium.PrimeSelenium;
import org.primefaces.selenium.component.base.AbstractInputComponent;

import java.util.List;

/**
 * Component wrapper for the PrimeFaces {@code p:menubar}.
 */
public abstract class Menubar extends AbstractInputComponent {

    public WebElement findMenuitemByValue(String value) {
        return findMenuitemByValue(this, value);
    }

    public WebElement findMenuitemByValue(WebElement parentElt, String value) {
        List<WebElement> subElements = parentElt.findElements(By.cssSelector("ul > li.ui-menuitem"));
        return subElements.stream()
                .filter(e -> e.getText().equals(value))
                .findFirst()
                .orElseThrow(() -> new NoSuchElementException("no menuitem with value '" + value + "'"));
    }

    /**
     * Select L1 menuitem
     * @param value
     * @return
     */
    public WebElement selectMenuitemByValue(String value) {
        Actions actions = new Actions(getWebDriver());
        actions.moveToElement(this.getWrappedElement()).build().perform(); // FF - workaround when clicking on multiple menuitems successively
        return selectMenuitemByValue(this, value);
    }

    /**
     * Select L2+ menuitem
     * @param parentElt parent menuitem
     * @param value
     * @return
     */
    public WebElement selectMenuitemByValue(WebElement parentElt, String value) {
        WebElement elt = findMenuitemByValue(parentElt, value);

        if ((PrimeSelenium.hasCssClass(elt, "ui-menu-parent")) && isToggleEventHover()) {
            Actions actions = new Actions(getWebDriver());
            actions.moveToElement(elt).build().perform();
        }
        else {
            WebElement eltA = elt.findElement(By.xpath("./a"));
            if (isAjaxified(eltA, "onclick")) {
                PrimeSelenium.guardAjax(elt).click();
            }
            else {
                elt.click();
            }
            // some more cases?
        }

        return  elt;
    }

    public boolean isToggleEventHover() {
        return "hover".equals(getWidgetConfiguration().getString("toggleEvent"));
    }

    public boolean isToggleEventClick() {
        return "click".equals(getWidgetConfiguration().getString("toggleEvent"));
    }
}
