/*
 * Copyright 2009-2024 PrimeTek.
 *
 * https://www.primefaces.org/lts/licenses/
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.primefaces.selenium.component;

import org.openqa.selenium.By;
import org.openqa.selenium.WebElement;
import org.primefaces.selenium.PrimeExpectedConditions;
import org.primefaces.selenium.PrimeSelenium;
import org.primefaces.selenium.component.base.AbstractInputComponent;
import org.primefaces.selenium.component.base.ComponentUtils;

/**
 * Component wrapper for the PrimeFaces {@code p:colorPicker}.
 */
public abstract class ColorPicker extends AbstractInputComponent {

    /**
     * Gets the color pickup popup panel.
     * @return the WebElement for the panel
     */
    public WebElement getPanel() {
        return getWebDriver().findElement(By.id("clr-picker"));
    }

    /**
     * Gets the popup trigger button.
     * @return the WebElement for the trigger button
     */
    public WebElement getTriggerButton() {
        return findElement(By.xpath("preceding-sibling::*"));
    }

    /**
     * Is this component AJAX enabled with "open"?
     *
     * @return true if AJAX enabled false if not
     */
    public boolean isOpenAjaxified() {
        return ComponentUtils.hasAjaxBehavior(getRoot(), "open");
    }

    /**
     * Is this component AJAX enabled with "close"?
     *
     * @return true if AJAX enabled false if not
     */
    public boolean isCloseAjaxified() {
        return ComponentUtils.hasAjaxBehavior(getRoot(), "close");
    }

    /**
     * Set the current color using the widget.
     *
     * @param color the Hex/RGB/HSL color
     */
    public void setColor(String color) {
        PrimeSelenium.executeScript(isOnchangeAjaxified(), getWidgetByIdScript() + ".setColor('" + color + "');");
    }

    /**
     * Gets the current color from the widget.
     *
     * @return the Hex/RGB/HSL color or null
     */
    public String getColor() {
        return PrimeSelenium.executeScript("return " + getWidgetByIdScript() + ".getColor();");
    }

    /**
     * Shows the overlay panel.
     *
     * @return the panel shown
     */
    public WebElement showPanel() {
        if (isEnabled()) {
            if (!getPanel().isDisplayed()) {
                PrimeSelenium.executeScript(isOpenAjaxified(), getWidgetByIdScript() + ".show()");
            }
            PrimeSelenium.waitGui().until(PrimeExpectedConditions.visibleAndAnimationComplete(getPanel()));
        }
        return getPanel();
    }

    /**
     * Hides the overlay panel.
     * @param revert true to revert to the default color
     */
    public void hidePanel(boolean revert) {
        if (isEnabled()) {
            if (getPanel().isDisplayed()) {
                PrimeSelenium.executeScript(isCloseAjaxified(), getWidgetByIdScript() + ".hide(" + Boolean.toString(revert) + ");");
            }
            PrimeSelenium.waitGui().until(PrimeExpectedConditions.invisibleAndAnimationComplete(getPanel()));
        }
    }
}
