/*
 * Copyright 2009-2024 PrimeTek.
 *
 * https://www.primefaces.org/lts/licenses/
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * Licensed under PrimeFaces Commercial License, Version 1.0 (the "License");
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.primefaces.selenium.component;

import java.util.List;
import java.util.stream.Collectors;

import org.openqa.selenium.By;
import org.openqa.selenium.WebElement;
import org.primefaces.selenium.PrimeExpectedConditions;
import org.primefaces.selenium.PrimeSelenium;
import org.primefaces.selenium.component.base.AbstractInputComponent;
import org.primefaces.selenium.component.base.ComponentUtils;
import org.primefaces.selenium.findby.FindByParentPartialId;

/**
 * Component wrapper for the PrimeFaces {@code p:cascadeSelect}.
 */
public abstract class CascadeSelect extends AbstractInputComponent {

    @FindByParentPartialId("_input")
    private WebElement input;

    @FindByParentPartialId(value = "_panel", searchFromRoot = true)
    private WebElement panel;

    /**
     * Is the input using AJAX "itemSelect" event?
     *
     * @return true if using AJAX for itemSelect
     */
    public boolean isItemSelectAjaxified() {
        return ComponentUtils.hasAjaxBehavior(getRoot(), "itemSelect");
    }

    /**
     * Either display the dropdown or select the item it if is already displayed.
     */
    public void toggleDropdown() {
        if (getPanel().isDisplayed()) {
            hide();
        }
        else {
            show();
        }
    }

    /**
     * Shows the CascadeSelect panel.
     */
    public void show() {
        WebElement panel = getPanel();
        if (isEnabled() && !panel.isDisplayed()) {
            PrimeSelenium.executeScript(getWidgetByIdScript() + ".show();");
            PrimeSelenium.waitGui().until(PrimeExpectedConditions.visibleAndAnimationComplete(panel));
        }
    }

    /**
     * Hides the CascadeSelect panel.
     */
    public void hide() {
        WebElement panel = getPanel();
        if (isEnabled() && panel.isDisplayed()) {
            PrimeSelenium.executeScript(getWidgetByIdScript() + ".hide();");
            PrimeSelenium.waitGui().until(PrimeExpectedConditions.invisibleAndAnimationComplete(panel));
        }
    }

    public void select(String label) {
        if (isSelected(label) || !isEnabled()) {
            return;
        }

        if (!getPanel().isDisplayed()) {
            toggleDropdown();
        }

        boolean isLeaf = false;
        for (WebElement element : getItems()) {
            if (element.getAttribute("data-label").equalsIgnoreCase(label)) {
                isLeaf = !PrimeSelenium.hasCssClass(element, "ui-cascadeselect-item-group");
                click(element);
                break;
            }
        }
    }

    public String getSelectedLabel() {
        return getLabel().getText();
    }

    public boolean isSelected(String label) {
        boolean result = false;
        try {
            result = getSelectedLabel().equalsIgnoreCase(label);
        }
        catch (Exception e) {
            // do nothing
        }
        return result;
    }

    public List<String> getLabels() {
        return getItems().stream()
                    .map(e -> e.getAttribute("data-label"))
                    .collect(Collectors.toList());
    }

    public List<String> getValues() {
        return getItems().stream()
                    .map(e -> e.getAttribute("data-value"))
                    .collect(Collectors.toList());
    }

    @Override
    public WebElement getInput() {
        return input;
    }

    public WebElement getLabel() {
        return findElement(By.className("ui-cascadeselect-label"));
    }

    public List<WebElement> getItems() {
        return getPanel().findElements(By.className("ui-cascadeselect-item"));
    }

    public List<WebElement> getLeafItems() {
        return getItems().stream().filter(i -> !PrimeSelenium.hasCssClass(i, "ui-cascadeselect-item-group")).collect(Collectors.toList());
    }

    public WebElement getPanel() {
        return panel;
    }

    protected void click(WebElement element) {
        if (!PrimeSelenium.hasCssClass(element, "ui-cascadeselect-item-group") && (isOnchangeAjaxified() || isItemSelectAjaxified())) {
            PrimeSelenium.guardAjax(element).click();
        }
        else {
            element.click();
        }
    }
}
